import { API_KEYS, API_ENDPOINTS } from '../config/api'

export interface JamendoTrack {
  id: string
  name: string
  artist_name: string
  audio: string // 音频 URL
  image: string // 专辑封面 URL
  duration: number
  tags: string[]
}

export class JamendoService {
  async searchTracks(
    query: string,
    tags: string[] = [],
    limit: number = 10,
  ): Promise<JamendoTrack[]> {
    // 简化搜索参数
    const queryParams = new URLSearchParams({
      client_id: API_KEYS.JAMENDO_CLIENT_ID,
      format: 'json',
      limit: limit.toString(),
      tags: tags.slice(0, 2).join('+'), // 只使用前两个标签
      include: 'musicinfo',
      orderby: 'popularity_total', // 按总体流行度排序
    })

    // 如果有搜索词，添加到参数中
    if (query.trim()) {
      queryParams.append('namesearch', query.split(' ')[0]) // 只使用第一个搜索词
    }

    try {
      const response = await fetch(`${API_ENDPOINTS.JAMENDO_BASE_URL}/tracks?${queryParams}`)
      if (!response.ok) {
        throw new Error(`Jamendo API error: ${response.statusText}`)
      }

      const data = await response.json()

      // 如果没有结果，尝试只用标签重新搜索
      if (data.results.length === 0 && query.trim()) {
        return this.searchTracks('', tags, limit)
      }

      return data.results.map((track: any) => ({
        id: track.id,
        name: track.name,
        artist_name: track.artist_name,
        audio: track.audio,
        image: track.image,
        duration: track.duration,
        tags: track.tags || [],
      }))
    } catch (error) {
      console.error('Error fetching tracks from Jamendo:', error)
      throw error
    }
  }

  // 将情绪映射到 Jamendo 的标签和音乐风格
  mapEmotionToSearchTerms(emotion: string, intensity: number): { query: string; tags: string[] } {
    const emotionMappings: Record<string, { query: string; tags: string[] }> = {
      Happy: {
        query: 'happy',
        tags: ['pop', 'happy'],
      },
      Sad: {
        query: 'sad',
        tags: ['ambient', 'classical'],
      },
      Angry: {
        query: 'rock',
        tags: ['rock', 'metal'],
      },
      Relaxed: {
        query: 'ambient',
        tags: ['ambient', 'chillout'],
      },
      Excited: {
        query: 'dance',
        tags: ['electronic', 'dance'],
      },
    }

    return emotionMappings[emotion] || { query: '', tags: ['pop', 'rock'] }
  }
}
