export interface UserPreferences {
  theme: 'light' | 'dark'
  volume: number
  autoplay: boolean
  language: string
}

export interface EmotionHistory {
  timestamp: number
  emotion: string
  intensity: number
  tags: string[]
  description: string
}

export interface PlayHistory {
  songId: string
  timestamp: number
  emotion: string
  rating?: number
}

class StorageService {
  private readonly PREFS_KEY = 'mood_music_preferences'
  private readonly EMOTION_HISTORY_KEY = 'mood_music_emotion_history'
  private readonly PLAY_HISTORY_KEY = 'mood_music_play_history'

  // 默认偏好设置
  private defaultPreferences: UserPreferences = {
    theme: 'light',
    volume: 0.7,
    autoplay: true,
    language: 'en'
  }

  // 获取用户偏好
  getUserPreferences(): UserPreferences {
    try {
      const stored = localStorage.getItem(this.PREFS_KEY)
      return stored ? { ...this.defaultPreferences, ...JSON.parse(stored) } : this.defaultPreferences
    } catch (error) {
      console.error('Error loading preferences:', error)
      return this.defaultPreferences
    }
  }

  // 保存用户偏好
  saveUserPreferences(preferences: Partial<UserPreferences>): void {
    try {
      const current = this.getUserPreferences()
      const updated = { ...current, ...preferences }
      localStorage.setItem(this.PREFS_KEY, JSON.stringify(updated))
    } catch (error) {
      console.error('Error saving preferences:', error)
    }
  }

  // 获取情绪历史
  getEmotionHistory(): EmotionHistory[] {
    try {
      const stored = localStorage.getItem(this.EMOTION_HISTORY_KEY)
      return stored ? JSON.parse(stored) : []
    } catch (error) {
      console.error('Error loading emotion history:', error)
      return []
    }
  }

  // 添加情绪记录
  addEmotionHistory(entry: Omit<EmotionHistory, 'timestamp'>): void {
    try {
      const history = this.getEmotionHistory()
      const newEntry = { ...entry, timestamp: Date.now() }
      history.push(newEntry)
      
      // 只保留最近100条记录
      if (history.length > 100) {
        history.shift()
      }
      
      localStorage.setItem(this.EMOTION_HISTORY_KEY, JSON.stringify(history))
    } catch (error) {
      console.error('Error saving emotion history:', error)
    }
  }

  // 获取播放历史
  getPlayHistory(): PlayHistory[] {
    try {
      const stored = localStorage.getItem(this.PLAY_HISTORY_KEY)
      return stored ? JSON.parse(stored) : []
    } catch (error) {
      console.error('Error loading play history:', error)
      return []
    }
  }

  // 添加播放记录
  addPlayHistory(entry: Omit<PlayHistory, 'timestamp'>): void {
    try {
      const history = this.getPlayHistory()
      const newEntry = { ...entry, timestamp: Date.now() }
      history.push(newEntry)
      
      // 只保留最近100条记录
      if (history.length > 100) {
        history.shift()
      }
      
      localStorage.setItem(this.PLAY_HISTORY_KEY, JSON.stringify(history))
    } catch (error) {
      console.error('Error saving play history:', error)
    }
  }

  // 清除所有数据
  clearAll(): void {
    try {
      localStorage.removeItem(this.PREFS_KEY)
      localStorage.removeItem(this.EMOTION_HISTORY_KEY)
      localStorage.removeItem(this.PLAY_HISTORY_KEY)
    } catch (error) {
      console.error('Error clearing storage:', error)
    }
  }
}

// 导出单例
export const storageService = new StorageService()
